// gdt_*: Segmenttabelle (Speicherverwaltung)

void gdt_set_gate(int num, unsigned long base, unsigned long limit,
                  unsigned char access, unsigned char gran) {
  /* Setup the descriptor base address */
  gdt[num].base_low = (base & 0xFFFF);            // 16 bits
  gdt[num].base_middle = (base >> 16) & 0xFF;     //  8 bits
  gdt[num].base_high = (base >> 24) & 0xFF;       //  8 bits

  /* Setup the descriptor limits */
  gdt[num].limit_low  = (limit & 0xFFFF);         // 16 bits
  gdt[num].limit_high = ((limit >> 16) & 0x0F);   //  4 bits

  /* Finally, set up the granularity and access flags */
  gdt[num].flags = gran & 0xF;
  gdt[num].access = access;
}

void gdt_install() {
  gp.limit = (sizeof(struct gdt_entry) * 6) - 1;
  gp.base = (int) &gdt;

  gdt_set_gate(0, 0, 0, 0, 0);    // NULL descriptor

  // code segment
  gdt_set_gate(1, 0, 0xFFFFFFFF, 0b10011010, 0b1100 /* 0xCF */);

  // data segment
  gdt_set_gate(2, 0, 0xFFFFFFFF, 0b10010010, 0b1100 /* 0xCF */);

  gdt_flush();
}

page_desc* fill_page_desc (page_desc *pd, unsigned int present,
  unsigned int writeable, unsigned int user_accessible,
  unsigned int dirty, unsigned int frame_addr) {

  memset (pd, 0, sizeof(pd));
  
  pd->present = present;
  pd->writeable = writeable;
  pd->user_accessible = user_accessible;
  pd->dirty = dirty;
  pd->frame_addr = frame_addr >> 12;   // right shift, 12 bits
  return pd;
};

page_table_desc* fill_page_table_desc (page_table_desc *ptd, 
  unsigned int present, unsigned int writeable, 
  unsigned int user_accessible, unsigned int frame_addr) {

  memset (ptd, 0, sizeof(ptd));
  
  ptd->present = present;
  ptd->writeable = writeable;
  ptd->user_accessible = user_accessible;
  ptd->frame_addr = frame_addr >> 12;   // right shift, 12 bits
  return ptd;
};

static void set_frame (unsigned int frame_addr) {
  unsigned int frame = frame_addr / PAGE_SIZE;
  unsigned int index  = INDEX_FROM_BIT  (frame);
  unsigned int offset = OFFSET_FROM_BIT (frame);
  ftable[index] |= (1 << offset);
}

static void clear_frame (unsigned int frame_addr) {
  unsigned int frame = frame_addr / PAGE_SIZE;
  unsigned int index  = INDEX_FROM_BIT  (frame);
  unsigned int offset = OFFSET_FROM_BIT (frame);
  ftable[index] &= ~(1 << offset);
}

static unsigned int test_frame (unsigned int frame_addr) {
  // returns true if frame is in use (false if frame is free)
  unsigned int frame = frame_addr / PAGE_SIZE;
  unsigned int index  = INDEX_FROM_BIT  (frame);
  unsigned int offset = OFFSET_FROM_BIT (frame);
  return ((ftable[index] & (1 << offset)) >> offset);
}

int request_new_frame () {
  unsigned int frameid;
  boolean found=false;
  for (frameid = 0; frameid < NUMBER_OF_FRAMES; frameid++) {
    if ( !test_frame (frameid*4096) ) {
      found=true;
      break;   // frame found
    };
  }
  if (found) {
    memset ((void*)PHYSICAL(frameid << 12), 0, PAGE_SIZE);
    set_frame (frameid*4096);
    free_frames--;
    return frameid;
  } else {
    return -1;
  }
};

void release_frame (unsigned int frameaddr) {
  if ( test_frame (frameaddr) ) {
    // only do work if frame is marked as used
    clear_frame (frameaddr);
    free_frames++;
  };
};

unsigned int pageno_to_frameno (unsigned int pageno) {
  unsigned int pdindex = pageno/1024;
  unsigned int ptindex = pageno%1024;
  if ( ! current_pd->ptds[pdindex].present ) {
    return -1;       // we don't have that page table
  } else {
    // get the page table
    page_table* pt = (page_table*)
      ( PHYSICAL(current_pd->ptds[pdindex].frame_addr << 12) );
    if ( pt->pds[ptindex].present ) {
      return pt->pds[ptindex].frame_addr;
    } else {
      return -1;     // we don't have that page
    };
  };    
};

unsigned int* request_new_page (int need_more_pages) {
  unsigned int newframeid = request_new_frame ();
  if (newframeid == -1) { return NULL; }  // exit if no frame was found
  unsigned int pageno = -1;
  for (unsigned int i=0xc0000; i<1024*1024; i++) {
    if ( pageno_to_frameno (i) == -1 ) {
      pageno = i;
      break;       // end loop, unmapped page was found
    };
  };

  if ( pageno == -1 ) {
    return NULL;   // we found no page -- whole 4 GB are mapped???
  };
  unsigned int pdindex = pageno/1024;
  unsigned int ptindex = pageno%1024;
  page_table* pt;
  if (ptindex == 0) {
    // last entry! // create a new page table in the reserved frame
    page_table* pt = (page_table*) PHYSICAL(newframeid<<12);
    memset (pt, 0, PAGE_SIZE);
    KMAPD ( &(current_pd->ptds[pdindex]), newframeid << 12 );

    newframeid = request_new_frame ();  // get yet another frame
    if (newframeid == -1) {
      return NULL;                      // exit if no frame was found
      // note: we're not removing the new page table since we assume
      // it will be used soon anyway
    }
  };
  pt = (page_table*)( PHYSICAL(current_pd->ptds[pdindex].frame_addr << 12) );
  // finally: enter the frame address
  KMAP ( &(pt->pds[ptindex]), newframeid * PAGE_SIZE );

  // invalidate cache entry
  asm volatile ("invlpg %0" : : "m"(*(char*)(pageno<<12)) );

  memset ((unsigned int*) (pageno*4096), 0, 4096);
  return ((unsigned int*) (pageno*4096));
}

void release_page (unsigned int pageno) {
  int frameno = pageno_to_frameno (pageno);  // we will need this later
  if ( frameno == -1 )  { return; }          // exit if no such page
  unsigned int pdindex = pageno/1024;
  unsigned int ptindex = pageno%1024;
  page_table* pt;
  pt = (page_table*)
    ( PHYSICAL(current_pd->ptds[pdindex].frame_addr << 12) );
  // write null page descriptor
  memset (&(pt->pds[ptindex]), 0, 4);
  fill_page_desc (&(pt->pds[ptindex]), false, false, false, false, 0);
  release_frame (frameno<<12);   // expects an address, not an ID
  asm volatile ("invlpg %0" : : "m"(*(char*)(pageno<<12)) );
  // gdt_flush ();
};

void init_memory() {
  for (int i=1; i<1024; i++) {
    fill_page_table_desc (&(current_pd->ptds[i]), false, false, false, 0);
  };
  KMAPD ( &(current_pd->ptds[  0]), (unsigned int)(current_pt)-0xC0000000 );
  KMAPD ( &(current_pd->ptds[768]), (unsigned int)(current_pt)-0xC0000000 );
  for (int i=0; i<1023; i++) {
    KMAP ( &(current_pt->pds[i]), i*4096 );
  };
  printf ("[2] page directory setup, with identity mapping\n");
  unsigned int cr0;
  char *kernel_pd_address;
  kernel_pd_address = (char*)(current_pd) - 0xC0000000;
  asm volatile ("mov %0, %%cr3" : : "r"(kernel_pd_address)); 
    // write CR3
  asm volatile ("mov %%cr0, %0" : "=r"(cr0) : );  // read  CR0
  cr0 |= (1<<31);      // Enable paging by setting PG bit 31 of CR0
  asm volatile ("mov %0, %%cr0" : : "r"(cr0) );   // write CR0
  printf ("[3] paging activated.\n");
  gdt_install ();  // replace "trick GDT" with regular GDT

  paging_ready = true;
  printf ("[4] regular GDT is active\n");
  
  memset (kernel_pt_ram, 0, 4);

  for (unsigned int fid=0; fid<NUMBER_OF_FRAMES; fid++) {
    KMAP ( &(kernel_pt_ram[fid/1024].pds[fid%1024]), fid*PAGE_SIZE );
  }
  unsigned int physaddr;
  for (int i=0; i<16; i++) {
    // get physical address of kernel_pt_ram[i]
    physaddr = (unsigned int)(&(kernel_pt_ram[i])) - 0xc0000000;
    KMAPD ( &(current_pd->ptds[832+i]), physaddr );
  };

  gdt_flush ();
  memset (ftable, 0, NUMBER_OF_FRAMES/8);  // all frames are free
  memset (ftable, 0xff, 128);
  free_frames -= 1024;

}


