/*  ulix.c
 *  Mini-Kernel, Code-Teile aus Ulix (http://www.ulixos.org/) bzw. 
 *  Vorabversionen davon entnommen.
 *  (c) 2008-2021 Felix Freiling und Hans-Georg Esser */

#include "ulix-types.h"
#include "ulix-constants.h"
#include "ulix-prototypes.h"
#include "ulix-globalvars.h"
#include "ulix-memory.c"
#include "ulix-helpers.c"

// Konstanten fuer Data und Status Port des Tastatur-Controllers
#define KBD_DATA_PORT   0x60 
#define KBD_STATUS_PORT 0x64

void fill_idt_entry (unsigned char num, unsigned long address, 
    unsigned short gdtsel, unsigned char flags, unsigned char type) {
  if (num >= 0 && num < 256) {
    idt[num].addr_low  = address & 0xFFFF; // address is the handler address
    idt[num].addr_high = (address >> 16) & 0xFFFF;
    idt[num].gdtsel    = gdtsel;           // GDT sel.: user mode or kernel mode?
    idt[num].zeroes    = 0;
    idt[num].flags     = flags;
    idt[num].type      = type;
  }
}

static void set_irqmask (unsigned short mask) {
  outportb (IO_PIC_MASTER_DATA, (char)(mask % 256) );
  outportb (IO_PIC_SLAVE_DATA,  (char)(mask >> 8)  );
}

unsigned short get_irqmask () {
  return inportb (IO_PIC_MASTER_DATA) 
      + (inportb (IO_PIC_SLAVE_DATA) << 8);
}

static void enable_interrupt (int number) {
  set_irqmask ( 
    get_irqmask ()        // the current value
    & ~(1 << number)      // 16 one-bits, but bit "number" cleared
  );
}

void irq_handler (struct regs *r) {
  int number = r->int_no - 32;                      // interrupt number
  void (*handler)(struct regs *r);                  // type of handler functions

  if (number >= 8)  
    outportb (IO_PIC_SLAVE_CMD, END_OF_INTERRUPT);  // notify slave  PIC
  outportb (IO_PIC_MASTER_CMD, END_OF_INTERRUPT);   // notify master PIC

  handler = interrupt_handlers[number];
  if (handler != NULL)  handler (r);
}

void install_interrupt_handler (int irq, void (*handler)(struct regs *r)) {
  if (irq >= 0 && irq < 16)
    interrupt_handlers[irq] = handler;
}

void fault_handler (struct regs *r) {
  if (r->int_no >= 0 && r->int_no < 32) {
    printf ("'%s' (%d) Exception at 0x%08x.\n", exception_messages[r->int_no], r->int_no, r->eip);        
    printf ("eflags: 0x%08x  errcode: 0x%08x\n", r->eflags, r->err_code);
    printf ("eax: %08x  ebx: %08x  ecx: %08x  edx: %08x \n", r->eax, r->ebx, r->ecx, r->edx);
    printf ("eip: %08x  esp: %08x  int: %8d  err: %8d \n", r->eip, r->esp, r->int_no, r->err_code);
    printf ("ebp: %08x  cs: %d  ds: %d  es: %d  fs: %d  ss: %x \n", r->ebp, r->cs, r->ds, r->es, r->fs, r->ss);
    printf ("System Stops\n");   
    asm ("cli; \n hlt;");
  }
}

int main () {
  posx = 0; posy = 8;  // set cursor
  printf ("[1] entering main()\n");
  init_memory();    // <- fuer Tastatur-Aufgabe nicht noetig.

  idtp.limit = (sizeof (struct idt_entry) * 256) - 1;   // must do -1
  idtp.base  = (int) &idt;
  idt_load ();
  // FILL_IDT ist Makro; geht nicht in Schleife
  FILL_IDT( 0); FILL_IDT( 1); FILL_IDT( 2); FILL_IDT( 3); FILL_IDT( 4); 
  FILL_IDT( 5); FILL_IDT( 6); FILL_IDT( 7); FILL_IDT( 8); FILL_IDT( 9);
  FILL_IDT(10); FILL_IDT(11); FILL_IDT(12); FILL_IDT(13); FILL_IDT(14); 
  FILL_IDT(15); FILL_IDT(16); FILL_IDT(17); FILL_IDT(18); FILL_IDT(19);
  FILL_IDT(20); FILL_IDT(21); FILL_IDT(22); FILL_IDT(23); FILL_IDT(24);
  FILL_IDT(25); FILL_IDT(26); FILL_IDT(27); FILL_IDT(28); FILL_IDT(29);
  FILL_IDT(30); FILL_IDT(31);

  // PIC1 und PIC2 programmieren
  outportb (IO_PIC_MASTER_CMD,  0x11);  // ICW1: initialize; begin programming
  outportb (IO_PIC_SLAVE_CMD,   0x11);  // ICW1: dito, for PIC2
  outportb (IO_PIC_MASTER_DATA, 0x20);  // ICW2 for PIC1: offset 0x20 
                                        // (remaps 0x00..0x07 -> 0x20..0x27)
  outportb (IO_PIC_SLAVE_DATA,  0x28);  // ICW2 for PIC2: offset 0x28 
                                        // (remaps 0x08..0x0f -> 0x28..0x2f)
  outportb (IO_PIC_MASTER_DATA, 0x04);  // ICW3 for PIC1: there's a slave on IRQ 2 
                                        // (0b00000100 = 0x04)
  outportb (IO_PIC_SLAVE_DATA,  0x02);  // ICW3 for PIC2: your slave ID is 2
  outportb (IO_PIC_MASTER_DATA, 0x01);  // ICW4 for PIC1 and PIC2: 8086 mode
  outportb (IO_PIC_SLAVE_DATA,  0x01);
  outportb (IO_PIC_MASTER_DATA, 0x00);  // PIC1: mask 0
  outportb (IO_PIC_SLAVE_DATA,  0x00);  // PIC2: mask 0
  set_irqmask (0xFFFF);           // initialize IRQ mask
  enable_interrupt (IRQ_SLAVE);   // IRQ slave

  void (*irqs[])() = { irq0, irq1, irq2,  irq3,  irq4,  irq5,  irq6,  irq7,
                       irq8, irq9, irq10, irq11, irq12, irq13, irq14, irq15 };
  for (int i=0; i<16; i++)
    fill_idt_entry (32+i, (unsigned int)irqs[i], 0x08, 0b1110, 0b1110);
  // flags: 1 (present), 11 (DPL 3), 0; type: 1110 (32 bit interrupt gate)

  // Platz fuer Code
  printf ("%s\n", "Hier koennte Ihr Test stehen.");

  for (;;);   // inifinite loop
}

